/************************************************************************************
Program Name: SinglePRCCS.do 
Description : Assign Single-Level Procedure CCS category array (PRCCS#) from Stata. 
Developed   : By Bob Houchens
Updated     : By David Ross on 10/27/2009.
            : By David Ross and Nils Nordstrand on 11/03/2016 incorporating updates from Rebecca Pope (StataCorp).
************************************************************************************/

version 9

local modern = c(stata_version) >= 14
local old = c(stata_version) < 11

local vv : di "version " string(_caller()) ":"
*
unab varlist: PR*
local _npr: word count `varlist'
*
// only reshape if there is more than one PR 
if `_npr' > 1 {
    tempvar i j
    gen `i' = _n 	// use -generate- instead of -egen- (faster)
    qui compress	// reshape is slow, but faster the smaller the dataset is
    reshape long PR, i(`i') j(`j') 
}
*
* Generate a temporary procedure variable that will be reformatted by the clean function in preparation for the merge
*
generate _PR = PR
*
* Check the validity of the procedure code
*
tempvar invalid
capture icd9 check _PR, generate(`invalid')
*
* Format the temporary procedure to match the format in SinglePRCCS.dta  
*
if `modern' {
    `vv' qui icd9p clean _PR, dots, if `invalid'==0 | `invalid'==10
}
*
* replace invalid temporary diagnoses in preparation for the clean function
* (only necessary before Stata 14)
*
else {
    qui replace _PR="0000" if `invalid' > 0 & `invalid' < 10
    qui icd9p clean _PR, dots
}
drop `invalid'
*
* Merge the CCS category variable, PRCCS, that matches the temporary procedure
*
if `old' {
    * Sort by formatted procedure.
    sort _PR
    merge _PR using SinglePRCCS, nokeep
    drop _merge _PR
}
else {
    `vv' merge m:1 _PR using SinglePRCCS, keep(1 3) nogen 
    capture drop _PR
}
*
* Drop temporary variables and put data in original shape if needed
*
if `_npr' > 1 {
    reshape wide PR PRCCS, i(`i') j(`j')
    capture drop `i' 
}
*
exit
