﻿/******************************************************************/
/* Title:       CHRONIC CONDITION INDICATOR REFINED FOR ICD-10-CM */
/*              MAPPING PROGRAM                                   */
/*                                                                */
/* Program:     CCIR_Mapping_Program_v2026-1.sas                  */
/*                                                                */
/* Diagnoses:   v2026-1 is compatible with ICD-10-CM diagnosis    */
/*              codes from October 2015 through September 2026.   */
/*              ICD-10-CM codes should not include embedded       */
/*              decimals (example: S0100XA, not S01.00XA).        */
/*                                                                */
/* Description: This SAS mapping program adds the                 */
/*              chronic condition indicator data elements to the  */
/*              user’s ICD-10-CM-coded data.                    */
/*                                                                */
/*              There are two general sections to this program:   */
/*                                                                */
/*              1) The first section creates a temporary SAS      */
/*                 format using the Chronic Condition Indicator   */
/*                 Refined for ICD-10-CM CSV file. This informat  */
/*                 is used to create the chronic condition        */
/*                 indicator variables                            */                                    
/*              2) The second section loops through the diagnosis */
/*                 array in your SAS dataset and assigns the      */
/*                 chronic condition indicator variables added to */
/*                 the output file.                               */
/*                                                                */
/* Output:      This program appends the chronic condition        */
/*              indicator tool to the input SAS file. The data    */
/*              elements are named CCIRn, where n ranges from 1   */
/*              to the maximum number of available diagnoses.     */
/*                                                                */
/******************************************************************/


/*******************************************************************/
/*      THE SAS MACRO FLAGS BELOW MUST BE UPDATED BY THE USER      */ 
/*  These macro variables must be set to define the locations,     */
/*  names, and characteristics of your input and output SAS        */
/*  formatted data.                                                */
/*******************************************************************/


/**************************************/
/* SPECIFY INPUT FILE CHARACTERISTICS */
/**************************************/ 
* Specify the prefix used to name the ICD-10-CM
  diagnosis data element array in the input 
  dataset. In this example, the diagnosis data 
  elements would be named I10_DX1, I10_DX2, etc., 
  similar to the naming of ICD-10-CM data elements  
  in the HCUP databases;                                                      %LET DXPREFIX = I10_DX;    *<===USER MUST MODIFY;
* Specify the maximum number of diagnosis codes
  on any record in the input file. ;                                          %LET NUMDX = 40;            *<===USER MUST MODIFY;
* Specify the name of the variable that contains a 
  count of the ICD-10-CM codes reported on a record.  
  If no such variable exists, leave macro blank;                              %LET NDXVAR = I10_NDX;            *<===USER MUST MODIFY;
* Set the number of observations to use from 
  the input dataset. Use MAX to use all observations
  and use a smaller value for testing the program.;                           %LET OBS = max;            *<===USER MAY MODIFY; 


/**************************************/
/*       SPECIFY FILE LOCATIONS       */
/**************************************/

FILENAME INRAW1  'c:\directory\CCIR_v2026-1.csv';     * Location of Chronic Condition Indicator Refined CSV  file. <===USER MUST MODIFY;
LIBNAME  IN1     'c:\sasdata\';                       * Location of input discharge data.                          <===USER MUST MODIFY;
LIBNAME  OUT1    'c:\sasdata\';                       * Location of output data.                                   <===USER MUST MODIFY;


/***************************************/
/* SPECIFY INPUT AND OUTPUT FILE NAMES */
/***************************************/  
* Specify the name of the input dataset;                        %LET CORE = YOUR_SAS_INPUT_FILE_HERE;           *<===USER MUST MODIFY;
* Specify the name of the output dataset;                       %LET OUT1 = OUTPUT_SAS_FILE;                    *<===USER MUST MODIFY;


/*********************************************/
/*   SET CCIR VERSION                        */
/*********************************************/ 
%LET CCIR_VERSION = "2026.1" ;                                   *<=== DO NOT MODIFY;

TITLE1 'Chronic Condition Indicator Refined for ICD-10-CM';
TITLE2 'Mapping Program';



/*******************  SECTION 1: CREATE INFORMATS   ****************/
/*  SAS Load the Chronic Condition Indicator Refined for ICD-10-CM */
/*  mapping file and convert it into a temporary SAS format that   */
/*  will be used to assign the chronic condition indicator         */
/*  variables in the next step.                                    */
/*******************************************************************/
data ccir;
   infile INRAW1 dsd dlm=',' firstobs=3 end=eof;
   input
      start       : $char7.    
      code_label  : $char224.
      label       : 3. 
      ;

   retain hlo " ";
   fmtname = "ccir" ;
   type    = "i" ;
   output ;
   if eof then do ;
      start = " " ;
      label = .   ;
      hlo   = "o" ;
      output ;
   end ;
run;

proc format lib=work cntlin = ccir;
run;


options obs=&obs.;


/*** SECTION 2: CREATE CHRONIC CONDITION INDICATOR Refined for ICD-10-CM Variables ***/
/*  Create Chronic Condition Indicator Refined for ICD-10-CM diagnosis codes using   */
/*  the SAS format created above and the SAS output file you specified. Users can    */
/*  change the names of the output chronic condition indicator variables if needed   */
/*  here. It is also important to make sure that the correct ICD-10-CM diagnosis     */
/*  code prefixes are specified correctly in the macro DXPREFIX above.               */                                                  
/*************************************************************************************/  
%macro ccir;
%if &NUMDX > 0 %then %do; 
options obs=max;

data out1.&OUT1. (drop = i);  
   label ccir_version = "Version of Chronic Condition Indicator Refined for ICD-10-CM";
   retain CCIR_VERSION &CCIR_VERSION;

   set in1.&CORE.; 

   /***************************************************/
   /*  Loop through the ICD-10-CM diagnosis code      */
   /*  array in your SAS dataset and create the       */
   /*  chronic condition indicator variables.         */
   /***************************************************/
   array ccirs (*)   3 CCIR1-CCIR&NUMDX;                            * Suggested name for the chronic condition indicator variables.        <===USER MAY MODIFY;
   array dxs   (*)   $ &DXPREFIX.1-&DXPREFIX.&NUMDX;                * Name of Diagnosis variables on your file using the prefix specified at the top of the program;        

   %if &NDXVAR ne %then %let MAXNDX = &NDXVAR;
   %else                %let MAXNDX = &NUMDX;

   do i = 1 to min(&MAXNDX,dim(dxs));
      ccirs(i) = input(dxs(i),ccir.);
   end;  
 
   %do i = 1 %to &NUMDX;
      label CCIR&i. = "ICD-10-CM Chronic Condition Indicator Refined &i.";    * Labels for CCIR variables                               <===USER MAY MODIFY;    
   %end;
run;

proc print data=out1.&OUT1. (obs=20);                            * Change the number of observations if you wish to display more or less than 10.   <=== USER MAY MODIFY;
   var &DXPREFIX.1-&DXPREFIX.&NUMDX CCIR1-CCIR&NUMDX.  ;
   TITLE3 "PARTIAL PRINT OF THE OUTPUT CHRONIC CONDITION INDICATOR REFINED FOR ICD-10-CM FILE";
run;
%end;
%else %do;
   %put;
   %put 'ERROR: NO DIAGNOSIS CODES SPECIFIED FOR MACRO VARIABLE NUMDX, PROGRAM ENDING';
   %put;
%end;

%mend ccir;
%ccir;
