/**********************************************************************/
/* Title:  PROCEDURE CLASSES REFINED FOR ICD-10-PCS MAPPING PROGRAM   */
/*                                                                	  */
/* Program:     pclass_icd10pcs_mapping_program_v2021-2.sas       	  */
/*                                                                    */
/* Description: This program assigns procedure classes to             */
/*              data with ICD-10-PCS procedure codes.                 */
/*                                                                    */
/*              There are two general sections to this program:       */
/*                                                                    */
/*              1) The first section creates temporary SAS        	  */
/*                 informats using the Procedure Classes Refined for  */
/*                 ICD-10-PCS tool file. The procedure classes are    */
/*                 located in columns 3 and 4. These informats are    */
/*                 used in step 2 to create the procedure class       */
/*                 variables.     	                                  */
/*              2) The second section loops through the procedure 	  */
/*                 arrays in your SAS dataset and assigns the     	  */
/*                 procedure classes.                             	  */
/*                                                                	  */
/**********************************************************************/


/*******************************************************************/
/*      THE SAS MACRO FLAGS BELOW MUST BE UPDATED BY THE USER      */ 
/*  These macro variables must be set to define the locations,     */
/*  names, and characteristics of your input and output SAS        */
/*  formatted data.                                                */
/*******************************************************************/

/**************************************/
/*          FILE LOCATIONS            */
/**************************************/       
FILENAME INRAW1  'Location of CSV file: PClass_ICD10PCS_v2021-2.csv' LRECL=300;  * Location of Procedure Classes Refined for ICD-10-PCS tool file. <===USER MUST MODIFY;
LIBNAME  IN1     'Location of input discharge data';                             * Location of the input dataset.                    <===USER MUST MODIFY;
LIBNAME  OUT1    'Directory to store output file';                               * Location of the output dataset.                   <===USER MUST MODIFY;
                                    
                                       
/**************************************/
/*            FILE NAMES              */
/**************************************/ 
* Specify the name of the input dataset;                     %LET CORE=YOUR_SAS_FILE_HERE;  *<===USER MUST MODIFY;
* Specify the name of the output dataset;                    %LET OUT1=OUTPUT_SAS_FILE;     *<===USER MUST MODIFY; 


/**************************************/
/*   INPUT FILE CHARACTERISTICS       */
/**************************************/ 
* Specify the prefix used to name the ICD-10-PCS 
  procedure data element array in the input dataset. 
  In this example the procedure data elements would be 
  named I10_PR1, I10_PR2, etc., similar to the naming 
  of ICD-10-PCS data elements in HCUP databases;             %LET PRPREFIX=I10_PR;          *<===USER MUST MODIFY;

* Specify the maximum number of procedure codes on 
  any record in the input file;                              %LET NUMPR=15;                 *<===USER MUST MODIFY;

* Specify the number of observations to use from the 
  input dataset.  Use MAX to use all observations and
  use a smaller value for testing the program;               %LET OBS = MAX;                *<===USER MAY MODIFY;


TITLE1 'CREATE ICD-10-PCS PROCEDURE CLASSES';
TITLE2 'USE WITH DISCHARGE ADMINISTRATIVE DATA THAT HAS ICD-10-PCS CODES';


/******************* SECTION 1: CREATE INFORMATS ******************/
/*  SAS Load the Procedure Classes Refined for ICD-10-PCS mapping */
/*  file and convert it into a temporary SAS informat that will   */
/*  be used to assign the procedure class fields in the next step.*/
/******************************************************************/
data pclass ;
    infile inraw1 dsd dlm=',' end = eof firstobs=3;
    input
       start            : $char7.
       icd10pcs_label   : $char100.
       label            : 3.
       pclass_label     : $char100.
    ;
   retain hlo " ";
   fmtname = "pclass" ;
   type    = "i" ;
   output;

   if eof then do ;
      start = " " ;
      label = . ;
      hlo   = "o";
      output ;
   end ;
run;

proc format lib=work cntlin = pclass ;
run;

/************** SECTION 2: CREATE REFINED PROCEDURE CLASSES ***********/
/*  Create procedure classes for ICD-10-PCS using the SAS             */
/*  informat created above & the SAS output dataset you specified.    */
/*  Users can change the names of the output procedure class          */
/*  variables if needed here. It is also important to make sure       */
/*  that the correct ICD-10-PCS procedure prefixes are specified      */
/*  correctly in the macro PRPREFIX above.                            */
/**********************************************************************/  
%macro pclass;
%if &numpr > 0 %then %do; 
options obs=&OBS.;

data out1.&OUT1. (DROP = I);
   set in1.&CORE;
   
   label pclass_version = "Version of ICD-10-PCS Procedure Classes Refined";
   retain PCLASS_VERSION "2021.2";

   /****************************************************/
   /* Loop through the PCS procedure array in your SAS */
   /* dataset & create the procedure class variables   */
   /* as well as the orproc flag.                      */
   /****************************************************/
   label orproc = "Indicates major (i.e., operating room) procedure found on the record";
   orproc = 0;

   array     pclass (*)  3 pclass1-pclass&NUMPR;           * Suggested name for procedure class variables.  <===USER MAY MODIFY;
   array prs        (*)  $ &PRPREFIX.1-&PRPREFIX.&NUMPR;           

   do i = 1 to &NUMPR;
      pclass(i) = input(prs(i), pclass.);  
      if pclass(i) in (3,4) then orproc=1;
   end;  
   %do i = 1 %to &NUMPR;
      label pclass&i. = "Procedure Class &i.";             * Labels for procedure class variables      <===USER MAY MODIFY;  
   %end;
run;

proc means data=out1.&OUT1. n nmiss mean min max;
   var pclass1-pclass&NUMPR. orproc;
   title2 "MEANS ON THE OUTPUT ICD-10-PCS PROCEDURE CLASSES";
run;
%end;
%else %do;
   %put;
   %put 'ERROR: NO PROCEDURE CODES SPECIFIED FOR MACRO VARIABLE NUMPR, PROGRAM ENDING';
   %put;
%end;

%mend pclass;
%pclass;