/******************************************************************/
/* Title: Surgery Flags for Services and Procedures Mapping       */
/*        Program                                                 */
/*                                                                */
/* Program: SurgeryFlags_Services_Procedures_Mapping_Program_     */
/*          v2019-2.sas                                            */
/*                                                                */
/* Description: This is the SAS mapping program to add the        */
/*              Surgery Flags for Services and Procedures to data */
/*              with HCPCS Level I codes, commonly referred to as */
/*              Current Procedural Terminology (CPT) codes.       */
/*              CPT is a proprietary coding system developed and  */
/*              maintained by the American Medical Association.   */
/*                                                                */
/*              There are two general sections to this program:   */ 
/*                                                                */
/*              1) The first section creates a temporary SAS      */
/*                 informat using the Surgery Flags-Services and  */
/*                 Procedures tool file. This informat is used in */
/*                 step 2 to create the surgery flags.            */
/*                                                                */
/*              2) The second section loops through the CPT       */
/*                 array in your SAS dataset and creates an array */
/*                 of surgery flags in the output file.           */ 
/*                 Values of each surgery flag will be: 2=Narrow, */
/*                 1=Broad, 0=Neither, or .=missing.              */
/*                                                                */
/*              Starting with v2019.2 of the Surgery Flags for    */
/*              Services and Procedures, the tool will assign     */
/*              flags to CPT codes that are in the following      */
/*              ranges:                                           */
/*              - CPT Category I, Surgery: 10004–69990            */
/*              - CPT Category I, Radiology: 70010–79999          */
/*              - CPT Category I, Medicine: 90281–99756,          */
/*                    excluding evaluation and management codes   */ 
/*                    in the range 99201-99499                    */
/*              - CPT Category III Codes, Temporary codes for     */
/*                    emerging or experimental services or        */
/*                    procedures: 0042T–0593T                     */
/*              Any CPT (or HCPCS) codes that are not in the      */
/*              above range are assigned a missing value for the  */
/*              surgery flag.                                     */
/*                                                                */
/*              The v2021-1 CSV file is compatible with CPT        */
/*              procedure codes valid on January, 2020, with the  */
/*              addition of code changes related to COVID-19      */
/*              which became effective between March and          */ 
/*              September 2020.                                   */
/*                                                                */
/*              The v2019-2 CSV file is compatible with CPT       */
/*              procedure codes valid at any time during calendar */
/*              years 2018 and 2019.                              */
/*                                                                */
/* Output: This program creates a horizontal array of surgery     */
/*         flags that have a one-to-one correspondence to the     */
/*         user-provided array of CPT codes. Users also have      */
/*         the option to create a flag to indicate the highest    */
/*         surgery flag value on each record.                     */
/*                                                                */
/******************************************************************/


/*******************************************************************/
/*      THE SAS MACRO FLAGS BELOW MUST BE UPDATED BY THE USER      */ 
/*  These macro variables must be set to define the locations,     */
/*  names, and characteristics of your input and output SAS        */
/*  formatted data.                                                */
/*******************************************************************/

/**************************************/
/*          FILE LOCATIONS            */
/**************************************/
FILENAME INRAW1  'c:\surgery flags\surgeryflags_services_procedures_2019-2.csv' LRECL=300;  * Location of Surgery Flags-Services and Procedures tool file. <===USER MUST MODIFY;
LIBNAME  IN1     'c:\sasdata\';                                                             * Location of input encounter-level data.                      <===USER MUST MODIFY;
LIBNAME  OUT1    'c:\sasdata\';                                                             * Location of encounter-level data.                            <===USER MUST MODIFY;


/**************************************/
/*            FILE NAMES              */
/**************************************/              
* Input SAS file name here;                     %LET CORE      = YOUR_SAS_INPUT_FILE_HERE;     *<===USER MUST MODIFY;
* Output SAS file member name;                  %LET SASOUT    = YOUR_SAS_OUTPUT_FILE_HERE;    *<===USER MUST MODIFY;


/**************************************/
/*   INPUT FILE CHARACTERISTICS       */
/**************************************/ 
* Maximum number of CPT codes on any record;    %LET NUMCPT    = 15;                           *<===USER MUST MODIFY;
* Set the number of observations to use from 
  your dataset (use MAX for all observations,
  other values for testing);                    %LET OBS       = MAX;                          *<===USER MAY MODIFY;    


/**************************************/
/*   OUTPUT FILE CHARACTERISTICS      */
/**************************************/ 
* Build optional record-level flag to indicate the highest surgery flag value on the record;    
                                                %LET RECORDLVL = 0;                            *<===USER MAY MODIFY; 



TITLE1 'CREATE SURGERY FLAGS-SERVICES AND PROCEDURES';
TITLE2 'USE WITH ADMINISTRATIVE DATA THAT HAVE CPT CODES'; 

options obs=max;

%Macro surgery_flags;
%if &numcpt > 0 %then %do; 
/******************* SECTION 1: CREATE INFORMATS  *******************/
/*  SAS Load the Surgery Flags-Services and Procedures tool and     */
/*  convert it into a temporary SAS informat used to assign the     */
/*  surgery flags for each  CPT code on a record.                   */
/********************************************************************/
data cpt_surgery_flags;
   infile INRAW1 dsd dlm=',' end = eof firstobs=3;
	input
	   range       : $char13.
	   label       : 1.
	;

   start = scan(range,1);
   end   = scan(range,2);

   retain hlo " ";
   fmtname = "cpsf" ;
   type    = "i" ;
   output;

   if eof then do ;
      start = " " ;
      end   = " " ;
      label = . ;
      hlo   = "o";
      output ;
   end ;
run;

proc format lib=work cntlin = cpt_surgery_flags;
run;

options obs=&obs.;
/******************* SECTION 2: CREATE SURGERY FLAGS ****************/
/*  Create a surgery flag for each CPT code using the SAS informat  */
/*  created above and the SAS file you wish to augment. Users can   */
/*  change the names of the output surgery flag if needed here. It  */
/*  is also important to make sure that the correct CPT code        */
/*  variable names from your SAS file are used in the array 'cpt'.  */
/********************************************************************/  
data out1.&SASOUT (DROP = i );			    
   set in1.&CORE;

   /***************************************************/
   /*  First build arrays for all of the surgery      */
   /*  flag variables that are going to be created.   */
   /*  Also, build arrays that contain the CPT codes  */
   /*  on the input file. Initialize optional summary */
   /*  flag.                                          */
   /***************************************************/
   array surgflags  (*)   3 surg_cpt_flg1-surg_cpt_flg&numcpt;   * Name for the surgery flag variables           <===USER MAY MODIFY;
   array cpt        (*)   $5 cpt1-cpt&numcpt;                    * Name for CPT code variables in your file      <===USER MAY MODIFY;


   /***************************************************/
   /*  Loop through the CPT code array on your SAS    */
   /*  dataset and create a surgery flag variable for */
   /*  each CPT code. Create optional summary flags   */
   /*  as well. This step makes use of the CPT arrays */
   /*  above.                                         */
   /***************************************************/
   %if &RECORDLVL = 1 %then %do;
      label recordlvl = "Indicates the highest surgery flag value on the record";
      recordlvl = 0;
   %end;
   do i = 1 to &NUMCPT;
      surgflags(i) = input(cpt (i),cpsf.);

      %if &RECORDLVL = 1 %then %do;   
         if surgflags(i) > recordlvl then do;
            recordlvl = surgflags(i);
         end;
      %end;

     %do i = 1 %to &NUMCPT;
        label surg_cpt_flg&i. = "Surgery Flag-Services and Procedures Flag &i.";      * Labels for Surgery Flag Variables      <===USER MAY MODIFY;    
     %end;
   end; 
run;

proc print data=out1.&SASOUT (obs=10);  * Change the number of observations if you wish to display more or less than 10;
   var 
       cpt1-cpt&NUMCPT. surg_cpt_flg1-surg_cpt_flg&NUMCPT
       %if &RECORDLVL=1 %then %do;recordlvl %end;
   ;
   title3 "PARTIAL PRINT OF THE SURGERY FLAGS FILE";
run;
%end;
%else %do;
   %put ;
   %put 'ERROR: NO CPT CODES SPECIFIED FOR MACRO VARIABLE NUMCPT, PROGRAM ENDING';
   %put;
%end;

%Mend surgery_flags;
%surgery_flags ;



