/******************************************************************/
/* Title:       CCSR FOR ICD-10-PCS MAPPING PROGRAM               */
/*                                                                */
/* Program:     PRCCSR_Mapping_Program_v2021-1.SAS                */
/*                                                                */
/* Description: This is the SAS mapping program to apply 		  */
/*			  the CCSR to the users ICD-10-PCS-coded data.       */
/*			  The mapping program includes two options 	          */
/*			  for the file structure of CCSR output.              */
/*                                                                */
/*              There are two general sections to this program:   */
/*                                                                */
/*              1) The first section creates temporary SAS        */
/*                 informats using the PRCCSR CSV file.           */
/*                 These informats are used in step 2 to create   */
/*                 the CCSR variables in the output files.        */
/*              2) The second section loops through the procedure */
/*                 array in your SAS dataset and assigns          */
/*                 CCSR categories in the output files.           */
/*                                                                */
/* Output:                                                        */
/*                                                                */
/*           The user can select from two output file structures: */
/*              + Vertical file's layout: RECID PRCCSR PR_POSITION*/
/*                                        PR_VERSION              */
/*              + Horizontal file's layout: RECID PRCCSR_BBBNNN   */
/*                                          PR_VERSION            */
/*              where BBB=3-letter clinical domain abbreviation,  */
/*                     NNN=3-digit number                         */
/* Note: 3/10/2021 The coding of the horizontal file was corrected*/
/*                 to retain information on the position of the   */
/*                 procedure code.                                */
/******************************************************************/

/******************************************************************/
/*  Macro Variables that must be set to define the characteristics*/
/*  of your SAS discharge data. Change these values to match the  */
/*  number of ICD-10-PCS codes in your dataset. Change CORE       */
/*  to match the name of your dataset. Set PRPREFIX to be the     */
/*  fixed part of the ICD-10-PCS code variables. For example,     */ 
/*  if the ICD-10-PCS code variables in your SAS dataset are      */
/*  I10_PR1, I10_PR2, ... then set PRPREFIX to I10_PR.            */
/******************************************************************/

%Let obs_ = MAX;                                                           *<=== Use smaller number for testing;

* FILE LOCATIONS ;       
FILENAME INRAW1  'Location of CSV file, PRCCSR_v2021-1.CSV'   lRECL=3000;  *<=== USER MUST MODIFY;
LIBNAME  IN1     'Location of input discharge data';                       *<=== USER MUST MODIFY;
LIBNAME  OUT1    'Directory to store output file';                         *<=== USER MUST MODIFY;

* OPTIONAL FILES ;
* Build vertical file? 1=yes, 0=no;           %LET VERT=1;                          *<=== USER MUST MODIFY;
* Build horizontal file? 1=yes, 0=no;         %LET HORZ=1;                          *<=== USER MUST MODIFY;

* FILE NAMES ;
* Input SAS file member name;                 %LET CORE=INPUT_SAS_FILE;             *<=== USER MUST MODIFY;
* Output SAS file name, vertical;             %LET VERTFILE=OUTPUT_VERT_FILE_NAME;  *<=== USER MUST MODIFY; 
* Output SAS file name, horizontal;           %LET HORZFILE=OUTPUT_HORZ_FILE_NAME;  *<=== USER MUST MODIFY;

* INPUT DATA CHARACTERISTICS ;
* Linking key on input SAS data;              %LET RECID=KEY;                       *<=== USER MUST MODIFY; 
* Prefix of ICD-10-PCS procedure variables;   %LET PRPREFIX=I10_PR;                 *<=== USER MUST MODIFY;
* Maximum number of PRs on any record;        %LET NUMPR=25;                        *<=== USER MUST MODIFY;
* Record specific ICD-10-PCS Count variable, if not available leave it blank;            
                                              %LET NPRVAR=I10_NPR;                  *<=== USER MUST MODIFY;

* Facilitating macro used in multiple places; %LET PRCCSR_VERSION = "2021.1" ;      *<=== DO NOT MODIFY;

TITLE1 'CREATE ICD-10-PCS PRCCSR TOOL CATEGORIES';
TITLE2 'USE WITH DISCHARGE ADMINISTRATIVE DATA THAT HAS ICD-10-PCS CODES';

/******************* SECTION 1: CREATE INFORMATS ****************************/
/*  SAS Load the CCSR CSV file & convert into temporary SAS informats that  */
/*  will be used to assign the PRCCSR variables in the next step.           */
/****************************************************************************/
DATA PRCCSR;
    LENGTH LABEL $1140;
    INFILE INRAW1 DSD DLM=',' END = EOF FIRSTOBS=2;
    INPUT
       START             : $CHAR7.
       I10Label          : $CHAR124.
       I10CCSR           : $10.
       I10CCSRLabel      : $CHAR228.
       I10CCSRCD         : $CHAR228.
    ;

   RETAIN HLO " " FMTNAME "$PRCCSR" TYPE  "J" ;
   
   LABEL = I10CCSR;
   OUTPUT;

   IF EOF THEN DO ;
      START = " " ;
      LABEL = " " ;
      HLO   = "O";
      OUTPUT ;
   END ;
RUN;

PROC FORMAT LIB=WORK CNTLIN = PRCCSR;
RUN;

DATA PRCCSRL(KEEP=START LABEL FMTNAME TYPE HLO);
  SET PRCCSR(KEEP=I10CCSR: rename=(I10CCSR=START I10CCSRLabel=LABEL)) END=EOF;

  RETAIN HLO " " FMTNAME "$PRCCSRL" TYPE  "J" ;
  OUTPUT;
  
  IF EOF THEN DO ;
     START = " " ;
     LABEL = " " ;
     HLO   = "O";
     OUTPUT;
  END;
run;

PROC SORT DATA=PRCCSRL NODUPKEY; 
  BY START; 
RUN;

PROC FORMAT LIB=WORK CNTLIN = PRCCSRL;
RUN;

DATA PRCCSRCD(KEEP=START LABEL FMTNAME TYPE HLO);
  SET PRCCSR(KEEP=I10CCSR: rename=(I10CCSR=START I10CCSRCD=LABEL)) END=EOF;

  RETAIN HLO " " FMTNAME "$PRCCSRCD" TYPE  "J" ;
  OUTPUT;
  IF EOF THEN DO ;
     START = " " ;
     LABEL = " " ;
     HLO   = "O";
     OUTPUT;
  END;
run;

PROC SORT DATA=PRCCSRCD NODUPKEY; BY START; RUN;

PROC FORMAT LIB=WORK CNTLIN = PRCCSRCD;
RUN;

/*********** SECTION 2: CREATE ICD-10-PCS CCSR OUTPUT FILES **********************/
/*  Create CCSR categories for ICD-10-PCS using the SAS informats created        */
/*  in Section 1 and the ICD-10_PCS codes in your SAS dataset.                   */
/*  At most two separate output files are created plus a few intermediate files  */
/*  for the construction of the horizontal file                                  */
/*********************************************************************************/  

%Macro prccsr_vt;
   DATA &VERTFILE (KEEP=&RECID PRCCSR PR_POSITION PRCCSR_VERSION) ;
     RETAIN &RECID;
     LENGTH ICD10_Code $7 PRCCSR $6 PR_POSITION 3 PRCCSR_VERSION $6;
     LABEL PRCCSR   = "CCSR category for ICD-10-PCS procedures"
		   PR_POSITION = "Position of code in input procedure array"
		   PRCCSR_VERSION = "Version of CCSR for ICD-10-PCS procedures"
		   ;
     RETAIN PRCCSR_VERSION &PRCCSR_VERSION;

     SET &CORE.Skinny;
	 by &RECID;
     array A_PR(&NUMPR)       &PRPREFIX.1-&PRPREFIX.&NUMPR;

     %if &NPRVAR ne %then %let MAXNPR = &NPRVAR;
     %else %let MAXNPR=&NUMPR;
 
     DO I=1 TO min(&MAXNPR,dim(A_PR));
       ICD10_CODE=A_PR(I);
       PR_POSITION=I;

       PRCCSR=INPUT(A_PR(I), $PRCCSR.); 
       if not missing(ICD10_CODE) and missing(PRCCSR) then do;
	      ***invalid ICD-10-PCS found;
		  PRCCSR='InvlPR';
	   end;
       if not missing(ICD10_CODE) then output &VERTFILE;
     end; 
run;
Title1 "Vertical file";
proc contents data=&VERTFILE varnum;
run;
Title2 "Sample print of vertical file";
proc print data=&VERTFILE(obs=10);
run;
%mend;

* =========================================================================== * 
* Count maximum number of PRCCSR values for each clinical domain.             *
* Please do not change this code. It is necessary to the program function.    *
* =========================================================================== *;
%macro count_prccsr;
  DATA ClinicalDomain;
    length cd cdnum $3 ;
    SET PRCCSR(KEEP=I10CCSR:) END=EOF;
    
    cd=substr(I10CCSR, 1, 3);
    cdnum=substr(I10CCSR, 4, 3);
    output;
    keep cd cdnum;
   run;
   proc sort data=ClinicalDomain; 
     by cd cdnum ; 
   run;
   data cd_max;
     set ClinicalDomain;
     by cd cdnum;
     if last.cd;
   run;
   %global mncd;
   %global cd_;
   proc sql noprint;
     select distinct cd into :cd_ separated by ' '
     from cd_max
     ; 
   quit;
   data null;
     set cd_max end=eof;
     if eof then call symput("mncd", put(_N_, 2.)); 
   run; 

   %do i=1 %to &mncd;
     %let cd=%scan(&cd_, &i);
     %global max&cd. ;
   %end;  

   data null;
     set cd_max end=eof;
     mcd="max" || cd; 
     call symput(mcd, cdnum); 
   run; 

   %put verify macro definition:;
   %put mncd=&mncd;
   %do i=1 %to &mncd;
     %let cd=%scan(&cd_, &i);
     %put max&cd._ = &&&max&cd;
   %end;  
%mend;

%macro prccsr_hz;
* =========================================================================== * 
* Create horizontal file layout using vertical file                           *
* =========================================================================== *;
Data PRCCSR_First(keep=&RECID PRCCSR) PRCCSR_second(keep=&RECID PRCCSR);
  set &VERTFILE;
  by &RECID;
  if PRCCSR not in ('InvlPR');
  if PR_Position = 1 then output PRCCSR_First;
  else output PRCCSR_Second;
run;

proc sort data=PRCCSR_second nodupkey;
  by &RECID PRCCSR;
run;
proc sort data=PRCCSR_First;
  by &RECID PRCCSR;
run;

data PRCCSR;
  length PR_Position 3;
  merge PRCCSR_First(in=inp) PRCCSR_Second(in=ins);
  by &RECID PRCCSR;
  if inp and not ins then PR_Position = 1;
  else if ins and not inp then PR_Position = 3;
  else PR_Position = 2;
run;

proc transpose data=PRCCSR out=PRCCSR_Transposed(drop=_NAME_) prefix=PRCCSR_; 
  by &RECID;
  ID PRCCSR;
  Var PR_Position;
run; 

**** Some input records may not have any or only invalid ICD-10_PCS codes 
     and not be represented in the vertical file.
     Ensure the horizontal output file has the same number of records as input file;
data out1.&HORZFILE ;
  retain &RECID; 
  LENGTH PRCCSR_VERSION $6
    %do i=1 %to &mncd; 
      %let cd=%scan(&cd_, &i);
      PRCCSR_&cd.001-PRCCSR_&cd.&&max&cd. 
    %end;
    3 ;
   Label PRCCSR_VERSION = "Version of CCSR for ICD-10-PCS procedures" 
     %do i=1 %to &mncd; 
      %let cd=%scan(&cd_, &i);
	  %do j=1 %to &&max&cd. ;
	     %if &j < 10 %then PRCCSR_&cd.00&j = "Indication that at least one ICD-10-PCS procedure on the record is included in CCSR &cd.00&j" ;
	     %else %if &j < 100 %then PRCCSR_&cd.0&j = "Indication that at least one ICD-10-PCS procedure on the record is included in CCSR &cd.0&j" ;
	     %else PRCCSR_&cd.&j = "Indication that at least one ICD-10-PCS procedure on the record is included in CCSR &cd.&j" ;
	  %end;
    %end;
    ;
  RETAIN PRCCSR_VERSION &PRCCSR_VERSION;
  
  merge &CORE.Skinny(in=ini keep=&RECID) PRCCSR_Transposed;
  by &RECID;

  if not ini then abort; 

  ***If no ICD-10-PCS are found on the record, or any PRCCSR not exists on records, set PRCCSR_* values to 0;
  array a _numeric_;
  do over a;
    if a = . then a=0;
  end;
run;

Title1 "Horizontal file";
proc contents data=out1.&HORZFILE varnum;
run;
Title2 "Sample print of horizontal file";
proc print data=out1.&HORZFILE(obs=10);
run;
%mend;

%macro main;
   %count_prccsr;
   proc sort data=IN1.&CORE(obs = &obs_ keep=&RECID &NPRVAR &PRPREFIX.1-&PRPREFIX.&NUMPR) out=&CORE.Skinny; by &RECID;  run;
   %prccsr_vt;
   %if &vert = 1 %then %do; 
	 data OUT1.&VERTFILE(SortedBy=&RECID);
	   set &VERTFILE;
	   by &RECID;
	 run;  
   %end;
   %if &horz = 1 %then %do; 
     %prccsr_hz;
   %end;
%mend;
%main;

